const fs = require('fs');
const path = require('path');
const { exec } = require('child_process');
const csv = require('csv-parser');

const INPUT_CSV = 'searchx-list.csv'; // file CSV input besar
const OUTPUT_DIR = 'batches';                // direktori untuk menyimpan CSV batch
const BATCH_SIZE = 5;                       // jumlah baris maksimum per batch
const MAX_PARALLEL = 10;                      // jumlah batch maksimal dijalankan paralel

// Membaca CSV lengkap
function readCSV(file) {
  return new Promise((resolve, reject) => {
    const results = [];
    fs.createReadStream(file)
      .pipe(csv())
      .on('data', (data) => results.push(data))
      .on('end', () => resolve(results))
      .on('error', reject);
  });
}

// Membuat file batch CSV dari list data
function writeBatchFiles(records) {
  if (!fs.existsSync(OUTPUT_DIR)) {
    fs.mkdirSync(OUTPUT_DIR);
  }

  let batchFiles = [];
  let batch = [];
  let batchIndex = 1;

  for (const record of records) {
    batch.push(record);
    if (batch.length >= BATCH_SIZE) {
      const batchFile = path.join(OUTPUT_DIR, `batch_${batchIndex}.csv`);
      writeCSVFile(batchFile, batch);
      batchFiles.push(batchFile);
      batchIndex++;
      batch = [];
    }
  }

  // Jika ada sisa batch
  if (batch.length > 0) {
    const batchFile = path.join(OUTPUT_DIR, `batch_${batchIndex}.csv`);
    writeCSVFile(batchFile, batch);
    batchFiles.push(batchFile);
  }
  return batchFiles;
}

// Menulis CSV file dari array record (object)
function writeCSVFile(file, records) {
  const headers = Object.keys(records[0]);
  const csvContent = [
    headers.join(','),
    ...records.map(r => headers.map(h => `"${(r[h]||'').replace(/"/g, '""')}"`).join(','))
  ].join('\n');
  
  fs.writeFileSync(file, csvContent);
}

// Eksekusi batch secara paralel dengan max concurrency
function runBatchesParallel(batchFiles) {
  return new Promise(resolve => {
    let running = 0, index = 0;
    const results = [];

    function runNext() {
      if (index >= batchFiles.length && running === 0) {
        resolve(results);
        return;
      }

      while (running < MAX_PARALLEL && index < batchFiles.length) {
        const file = batchFiles[index];
        index++;
        running++;

        console.log(`Starting batch: ${file}`);

        const child = exec(`node search-replace.js --file=${file}`, (error, stdout, stderr) => {
          running--;
          if (error) {
            console.error(`Batch ${file} failed:`, error);
            results.push({file, success: false, error});
          } else {
            console.log(`Batch ${file} completed.`);
            results.push({file, success: true});
          }
          runNext();
        });

        child.stdout.pipe(process.stdout);
        child.stderr.pipe(process.stderr);
      }
    }

    runNext();
  });
}

async function main() {
  console.log('Reading full CSV...');
  const records = await readCSV(INPUT_CSV);
  console.log(`Total records: ${records.length}`);

  console.log('Splitting into batches...');
  const batchFiles = writeBatchFiles(records);

  console.log(`Split into ${batchFiles.length} batches.`);

  console.log(`Running batches with max concurrency: ${MAX_PARALLEL}...`);
  const results = await runBatchesParallel(batchFiles);

  // Kesimpulan
  const successCount = results.filter(r => r.success).length;
  const failedCount = results.length - successCount;
  console.log(`\nAll batches completed. Success: ${successCount}, Failed: ${failedCount}`);
}

main().catch(console.error);
