const fs = require('fs');
const path = require('path');
const csv = require('csv-parser');
const argv = require('minimist')(process.argv.slice(2));

// Ambil argumen file CSV dari CLI dan update CONFIG
const replaceListFile = argv.file || 'search-replace-list.csv';

// Configuration
const CONFIG = {
  replaceListFile,        // File CSV search-replace, bisa di-override dari CLI
  sourceFolder: './content',
  fileExtensions: ['.html'],
  
  maxConcurrency: 10,     // Jumlah folder diproses paralel per batch
};

// Read CSV list
function readReplaceList() {
  return new Promise((resolve, reject) => {
    const results = [];
    fs.createReadStream(CONFIG.replaceListFile)
      .pipe(csv())
      .on('data', (data) => results.push(data))
      .on('end', () => resolve(results))
      .on('error', reject);
  });
}

// Group replacements by folder
function groupByFolder(replaceList) {
  const grouped = {};
  
  replaceList.forEach(item => {
    const folder = item.folder_path;
    if (!grouped[folder]) {
      grouped[folder] = [];
    }
    grouped[folder].push({
      search: item.search,
      replace: item.replace
    });
  });
  
  return grouped;
}

// Get all files with specific extensions recursively
function getAllFiles(dirPath, arrayOfFiles = []) {
  if (!fs.existsSync(dirPath)) {
    return arrayOfFiles;
  }
  
  const files = fs.readdirSync(dirPath);
  
  files.forEach(file => {
    const filePath = path.join(dirPath, file);
    
    if (fs.statSync(filePath).isDirectory()) {
      arrayOfFiles = getAllFiles(filePath, arrayOfFiles);
    } else {
      const ext = path.extname(file).toLowerCase();
      if (CONFIG.fileExtensions.includes(ext)) {
        arrayOfFiles.push(filePath);
      }
    }
  });
  
  return arrayOfFiles;
}

// Replace content in file
function replaceInFile(filePath, searchPattern, replacePattern) {
  try {
    let content = fs.readFileSync(filePath, 'utf-8');
    const originalContent = content;
    
    const regex = new RegExp(searchPattern.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'), 'g');
    const matches = content.match(regex);
    const count = matches ? matches.length : 0;
    
    content = content.replace(regex, replacePattern);
    
    if (content !== originalContent) {
      fs.writeFileSync(filePath, content, 'utf-8');
      return { success: true, count };
    }
    
    return { success: true, count: 0 };
  } catch (error) {
    return { success: false, error: error.message, count: 0 };
  }
}

// Process single folder with multiple replacements (ASYNC)
async function processFolder(folderPath, replacements, index, total) {
  const startTime = Date.now();
  const fullPath = path.join(CONFIG.sourceFolder, folderPath);
  
  console.log(`\n[${index + 1}/${total}] 📁 ${folderPath}`);
  
  const result = {
    folderPath,
    totalReplacements: 0,
    success: false,
    error: null,
    duration: 0,
    filesProcessed: 0
  };
  
  if (!fs.existsSync(fullPath)) {
    result.error = 'Folder not found';
    result.duration = ((Date.now() - startTime) / 1000).toFixed(2);
    console.log(`   ✗ Not found (${result.duration}s)`);
    return result;
  }
  
  const files = getAllFiles(fullPath);
  
  if (files.length === 0) {
    result.success = true;
    result.duration = ((Date.now() - startTime) / 1000).toFixed(2);
    console.log(`   - No files (${result.duration}s)`);
    return result;
  }
  
  let totalReplacements = 0;
  let filesWithChanges = 0;
  
  replacements.forEach((replacement) => {
    files.forEach(filePath => {
      const replaceResult = replaceInFile(filePath, replacement.search, replacement.replace);
      
      if (replaceResult.success && replaceResult.count > 0) {
        totalReplacements += replaceResult.count;
      }
    });
  });
  
  result.totalReplacements = totalReplacements;
  result.filesProcessed = files.length;
  result.success = true;
  result.duration = ((Date.now() - startTime) / 1000).toFixed(2);
  
  if (totalReplacements > 0) {
    console.log(`   ✓ ${totalReplacements} replacements in ${files.length} files (${result.duration}s)`);
  } else {
    console.log(`   - No matches in ${files.length} files (${result.duration}s)`);
  }
  
  return result;
}

// Process folders in batches with concurrency control
async function processFoldersInParallel(groupedReplacements) {
  const folders = Object.keys(groupedReplacements);
  const total = folders.length;
  const results = [];
  
  for (let i = 0; i < folders.length; i += CONFIG.maxConcurrency) {
    const batch = folders.slice(i, i + CONFIG.maxConcurrency);
    
    console.log(`\n━━━ Processing batch ${Math.floor(i / CONFIG.maxConcurrency) + 1} (${batch.length} folders) ━━━`);
    
    const batchPromises = batch.map((folder, batchIndex) => 
      processFolder(
        folder, 
        groupedReplacements[folder],
        i + batchIndex,
        total
      )
    );
    
    const batchResults = await Promise.all(batchPromises);
    results.push(...batchResults);
    
    if (i + CONFIG.maxConcurrency < folders.length) {
      await new Promise(resolve => setTimeout(resolve, 100));
    }
  }
  
  return results;
}

// Main execution
async function main() {
  console.log('╔═══════════════════════════════════════════╗');
  console.log('║  Search & Replace (Parallel Processing)  ║');
  console.log('╚═══════════════════════════════════════════╝\n');
  
  const startTime = Date.now();
  
  if (!fs.existsSync(CONFIG.replaceListFile)) {
    console.error(`✗ Replace list not found: ${CONFIG.replaceListFile}`);
    process.exit(1);
  }
  
  if (!fs.existsSync(CONFIG.sourceFolder)) {
    console.error(`✗ Content folder not found: ${CONFIG.sourceFolder}`);
    process.exit(1);
  }
  
  console.log(`Reading: ${CONFIG.replaceListFile}`);
  const replaceList = await readReplaceList();
  const groupedReplacements = groupByFolder(replaceList);
  
  const folderCount = Object.keys(groupedReplacements).length;
  const totalReplaceCount = replaceList.length;
  
  console.log(`Found ${folderCount} folder(s) with ${totalReplaceCount} replacement(s)`);
  console.log(`Concurrency: ${CONFIG.maxConcurrency} folders at once`);
  
  const results = await processFoldersInParallel(groupedReplacements);
  
  const totalTime = ((Date.now() - startTime) / 1000).toFixed(2);
  const successCount = results.filter(r => r.success).length;
  const failedCount = results.filter(r => !r.success).length;
  const totalReplacements = results.reduce((sum, r) => sum + r.totalReplacements, 0);
  const totalFiles = results.reduce((sum, r) => sum + r.filesProcessed, 0);
  
  console.log('\n╔═══════════════════════════════════════════╗');
  console.log('║                  SUMMARY                  ║');
  console.log('╚═══════════════════════════════════════════╝');
  console.log(`Total Folders: ${results.length}`);
  console.log(`  Success: ${successCount} ✓`);
  console.log(`  Failed: ${failedCount} ✗`);
  console.log(`Total Files Processed: ${totalFiles}`);
  console.log(`Total Replacements: ${totalReplacements}`);
  console.log(`Total Duration: ${totalTime}s`);
  console.log(`Average per Folder: ${(totalTime / results.length).toFixed(2)}s\n`);
}

main().catch(console.error);
