<?php
/**
 * SQLite Database Wrapper
 */

class Database {
    private $db;
    private $lastError;
    
    /**
     * Constructor - open database connection
     */
    public function __construct(string $dbPath) {
        if (!file_exists($dbPath)) {
            $this->lastError = "Database file not found: {$dbPath}";
            throw new Exception($this->lastError);
        }
        
        $this->db = new SQLite3($dbPath, SQLITE3_OPEN_READONLY);
        if (!$this->db) {
            $this->lastError = "Failed to open database: {$dbPath}";
            throw new Exception($this->lastError);
        }
    }
    
    /**
     * Destructor - close database connection
     */
    public function __destruct() {
        if ($this->db) {
            $this->db->close();
        }
    }
    
    /**
     * Get last error message
     */
    public function getLastError(): string {
        return $this->lastError ?: ($this->db ? $this->db->lastErrorMsg() : 'Unknown error');
    }
    
    /**
     * Execute query and return all results
     */
    public function queryAll(string $sql): array {
        $results = $this->db->query($sql);
        if (!$results) {
            $this->lastError = $this->db->lastErrorMsg();
            return [];
        }
        
        $rows = [];
        while ($row = $results->fetchArray(SQLITE3_ASSOC)) {
            $rows[] = $row;
        }
        
        return $rows;
    }
    
    /**
     * Get single row
     */
    public function queryOne(string $sql): ?array {
        $results = $this->db->query($sql);
        if (!$results) {
            $this->lastError = $this->db->lastErrorMsg();
            return null;
        }
        
        $row = $results->fetchArray(SQLITE3_ASSOC);
        return $row ?: null;
    }
    
    /**
     * Check if table exists
     */
    public function tableExists(string $tableName): bool {
        $sql = "SELECT name FROM sqlite_master WHERE type='table' AND name='{$tableName}'";
        $result = $this->queryOne($sql);
        return $result !== null;
    }
    
    /**
     * Get table count
     */
    public function getTableCount(string $tableName): int {
        $sql = "SELECT COUNT(*) as cnt FROM {$tableName}";
        $result = $this->queryOne($sql);
        return $result['cnt'] ?? 0;
    }
}
