const fs = require('fs');
const { execSync } = require('child_process');
const path = require('path');
const csv = require('csv-parser');

// Configuration
const CONFIG = {
  projectId: 'static-0001',
  mappingFile: 'bucket-mapping.csv',          // Di root folder
  sourceFolder: './content',                  // Di root folder
  reportFolder: './report',                   // Folder khusus untuk report
  reportFile: '',                             // Will be generated
  location: 'us-central1'
};

// Generate report filename
CONFIG.reportFile = path.join(CONFIG.reportFolder, `deployment-report-${Date.now()}.csv`);

// Execute gcloud command
function executeCommand(command, silent = false) {
  try {
    const output = execSync(command, { encoding: 'utf-8', stdio: silent ? 'pipe' : 'inherit' });
    return { success: true, output };
  } catch (error) {
    return { success: false, error: error.message };
  }
}

// Read CSV mapping
function readMapping() {
  return new Promise((resolve, reject) => {
    const results = [];
    fs.createReadStream(CONFIG.mappingFile)
      .pipe(csv())
      .on('data', (data) => results.push(data))
      .on('end', () => resolve(results))
      .on('error', reject);
  });
}

// Create bucket with public access
function createBucket(bucketName) {
  console.log(`\n[CREATE] ${bucketName}`);
  
  const createCmd = `gcloud storage buckets create gs://${bucketName} --project=${CONFIG.projectId} --location=${CONFIG.location} --uniform-bucket-level-access`;
  const createResult = executeCommand(createCmd, true);
  
  if (!createResult.success) {
    return { success: false, step: 'create', error: createResult.error };
  }
  
  const publicCmd = `gcloud storage buckets add-iam-policy-binding gs://${bucketName} --member=allUsers --role=roles/storage.objectViewer`;
  const publicResult = executeCommand(publicCmd, true);
  
  if (!publicResult.success) {
    return { success: false, step: 'set_public', error: publicResult.error };
  }
  
  console.log('✓ Bucket created and set to public');
  return { success: true };
}

// Upload content to bucket
function uploadContent(bucketName, folderName) {
  console.log(`[UPLOAD] ${folderName} → gs://${bucketName}/`);
  
  const folderPath = path.join(CONFIG.sourceFolder, folderName);
  
  if (!fs.existsSync(folderPath)) {
    return { success: false, error: `Folder not found: ${folderPath}` };
  }
  
  const uploadCmd = `gcloud storage cp -r "${folderPath}"/* gs://${bucketName}/`;
  const result = executeCommand(uploadCmd, true);
  
  if (result.success) {
    console.log('✓ Content uploaded successfully');
  }
  
  return result;
}

// Generate CSV report
function generateReport(results) {
  console.log('\n=== Generating Report ===');
  
  // Check if report folder exists
  if (!fs.existsSync(CONFIG.reportFolder)) {
    console.log('✗ Report folder does not exist. Creating folder...');
    fs.mkdirSync(CONFIG.reportFolder, { recursive: true });
  }
  
  const headers = [
    'Bucket Name',
    'Folder Name',
    'Status',
    'Bucket Created',
    'Public Access',
    'Content Uploaded',
    'Public URL',
    'Error Message',
    'Duration (s)',
    'Timestamp'
  ];
  
  let csvContent = headers.join(',') + '\n';
  
  results.forEach(result => {
    const row = [
      result.bucketName,
      result.folderName,
      result.success ? 'SUCCESS' : 'FAILED',
      result.bucketCreated ? 'YES' : 'NO',
      result.publicAccessSet ? 'YES' : 'NO',
      result.contentUploaded ? 'YES' : 'NO',
      result.success ? `https://storage.googleapis.com/${result.bucketName}/` : 'N/A',
      result.error ? `"${result.error.replace(/"/g, '""')}"` : '',
      result.duration || 0,
      new Date().toISOString()
    ];
    csvContent += row.join(',') + '\n';
  });
  
  fs.writeFileSync(CONFIG.reportFile, csvContent);
  console.log(`✓ Report saved: ${CONFIG.reportFile}`);
}

// Main execution
async function main() {
  console.log('╔═══════════════════════════════════════════╗');
  console.log('║   GCS Bulk Deployment with CSV Mapping   ║');
  console.log('╚═══════════════════════════════════════════╝\n');
  
  const startTime = Date.now();
  
  // Check if mapping file exists
  if (!fs.existsSync(CONFIG.mappingFile)) {
    console.error(`✗ Mapping file not found: ${CONFIG.mappingFile}`);
    console.error('Please create bucket-mapping.csv in the same folder as this script.');
    process.exit(1);
  }
  
  // Check if content folder exists
  if (!fs.existsSync(CONFIG.sourceFolder)) {
    console.error(`✗ Content folder not found: ${CONFIG.sourceFolder}`);
    console.error('Please create ./content folder in the same folder as this script.');
    process.exit(1);
  }
  
  // Read mapping
  const mappings = await readMapping();
  console.log(`Found ${mappings.length} buckets to process\n`);
  
  const results = [];
  
  // Process each mapping
  for (let i = 0; i < mappings.length; i++) {
    const mapping = mappings[i];
    const itemStartTime = Date.now();
    
    console.log(`\n[${i + 1}/${mappings.length}] ━━━━━━━━━━━━━━━━━━━━━━━━━━━━`);
    console.log(`Bucket: ${mapping.bucket_name}`);
    console.log(`Folder: ${mapping.folder_name}`);
    
    const result = {
      bucketName: mapping.bucket_name,
      folderName: mapping.folder_name,
      success: false,
      bucketCreated: false,
      publicAccessSet: false,
      contentUploaded: false,
      error: null,
      duration: 0
    };
    
    // Create bucket
    const createResult = createBucket(mapping.bucket_name);
    if (!createResult.success) {
      result.error = `Failed at ${createResult.step}: ${createResult.error}`;
      result.duration = ((Date.now() - itemStartTime) / 1000).toFixed(2);
      results.push(result);
      console.log(`✗ FAILED: ${result.error}`);
      continue;
    }
    
    result.bucketCreated = true;
    result.publicAccessSet = true;
    
    // Upload content
    const uploadResult = uploadContent(mapping.bucket_name, mapping.folder_name);
    if (!uploadResult.success) {
      result.error = `Upload failed: ${uploadResult.error}`;
      result.duration = ((Date.now() - itemStartTime) / 1000).toFixed(2);
      results.push(result);
      console.log(`✗ FAILED: ${result.error}`);
      continue;
    }
    
    result.contentUploaded = true;
    result.success = true;
    result.duration = ((Date.now() - itemStartTime) / 1000).toFixed(2);
    
    console.log(`✓ SUCCESS (${result.duration}s)`);
    results.push(result);
  }
  
  // Generate report
  generateReport(results);
  
  // Summary
  const totalTime = ((Date.now() - startTime) / 1000).toFixed(2);
  const successCount = results.filter(r => r.success).length;
  
  console.log('\n╔═══════════════════════════════════════════╗');
  console.log('║              DEPLOYMENT SUMMARY           ║');
  console.log('╚═══════════════════════════════════════════╝');
  console.log(`Total Buckets: ${results.length}`);
  console.log(`Success: ${successCount} ✓`);
  console.log(`Failed: ${results.length - successCount} ✗`);
  console.log(`Total Duration: ${totalTime}s`);
  console.log(`Report: ${CONFIG.reportFile}\n`);
}

// Run
main().catch(console.error);
