<?php
/**
 * Utility functions for HTML Exporter
 */

class Util {
    /**
     * Clean article text from JSON-escaped format
     */
    public static function cleanArticleText(?string $text): string {
        if (empty($text)) {
            return '';
        }
        
        try {
            $decoded = json_decode($text);
            if (is_string($decoded)) {
                return $decoded->trim();
            }
            return trim($text);
        } catch (Exception $e) {
            return trim($text);
        }
    }
    
    /**
     * Capitalize first letter of each word
     */
    public static function capitalizeWords(string $str): string {
        return preg_replace_callback('/\b\w/', function($matches) {
            return strtoupper($matches[0]);
        }, $str);
    }
    
    /**
     * Generate Bing image URL from keyword
     */
    public static function generateBingImageUrl(string $keyword): string {
        $encoded = urlencode($keyword);
        return "https://tse1.mm.bing.net/th?q={$encoded}&w=250&h=250&c=7";
    }
    
    /**
     * Validate URL format (must be http:// or https://)
     */
    public static function isValidUrl(?string $url): bool {
        if (empty($url)) {
            return false;
        }
        return filter_var($url, FILTER_VALIDATE_URL) !== false
            && preg_match('/^https?:\/\//i', $url);
    }
    
    /**
     * Create safe filename from slug
     */
    public static function createSafeSlug(string $text): string {
        $slug = strtolower(trim($text));
        $slug = preg_replace('/[^a-z0-9\-]/', '', $slug);
        $slug = preg_replace('/-+/', '-', $slug);
        return $slug ?: 'post';
    }
    
    /**
     * Parse JSON data safely
     */
    public static function parseJson(?string $json, $default = []) {
        if (empty($json)) {
            return $default;
        }
        
        if (is_string($json)) {
            $data = json_decode($json, true);
            return is_array($data) ? $data : $default;
        }
        
        return is_array($json) ? $json : $default;
    }
    
    /**
     * Truncate text to specified length
     */
    public static function truncateText(string $text, int $length, string $suffix = '...'): string {
        if (strlen($text) <= $length) {
            return $text;
        }
        return substr($text, 0, $length) . $suffix;
    }
    
    /**
     * Get first sentence from text
     */
    public static function getFirstSentence(string $text): string {
        $sentences = preg_split('/(?<=\.)\s+/', $text, 2);
        return trim($sentences[0] ?? $text);
    }
    
    /**
     * Parse spintax syntax {option1|option2|option3}
     * Returns a random selection from each option group
     */
    public static function parseSpintax(string $text): string {
        // Keep parsing until no more spintax patterns exist
        while (preg_match('/\{[^}]+\}/', $text)) {
            $text = preg_replace_callback('/\{([^}]+)\}/', function($matches) {
                $options = explode('|', $matches[1]);
                $randomOption = $options[array_rand($options)];
                return $randomOption;
            }, $text);
        }
        return $text;
    }
}
