<?php
/**
 * Template Engine for HTML Exporter
 */

class Template {
    private $templatePath;
    private $config;
    private $injectPath;
    
    /**
     * Constructor
     */
    public function __construct(string $templateName, string $basePath) {
        $this->templatePath = $basePath . '/templates/' . $templateName;
        $this->injectPath = $basePath . '/inject';
        $this->loadConfig();
    }
    
    /**
     * Load template configuration
     */
    private function loadConfig(): void {
        $configFile = $this->templatePath . '/config.php';
        if (file_exists($configFile)) {
            $this->config = require $configFile;
        } else {
            $this->config = [
                'name' => 'Unknown',
                'description' => 'No configuration',
                'version' => '1.0',
                'author' => 'Unknown',
                'css' => [],
                'inject' => []
            ];
        }
    }
    
    /**
     * Get template name
     */
    public function getName(): string {
        return $this->config['name'] ?? 'Unknown';
    }
    
    /**
     * Get template configuration
     */
    public function getConfig(): array {
        return $this->config;
    }
    
    /**
     * Check if template is valid
     */
    public function isValid(): bool {
        return file_exists($this->templatePath . '/header.php')
            && file_exists($this->templatePath . '/footer.php');
    }
    
    /**
     * Get list of available templates
     */
    public static function getAvailableTemplates(string $basePath): array {
        $templatesPath = $basePath . '/templates';
        if (!is_dir($templatesPath)) {
            return [];
        }
        
        $templates = [];
        foreach (scandir($templatesPath) as $dir) {
            if ($dir === '.' || $dir === '..') continue;
            $templatePath = $templatesPath . '/' . $dir;
            if (is_dir($templatePath)) {
                $configFile = $templatePath . '/config.php';
                if (file_exists($configFile)) {
                    $config = require $configFile;
                    $templates[$dir] = $config['name'] ?? $dir;
                } else {
                    $templates[$dir] = $dir;
                }
            }
        }
        
        return $templates;
    }
    
    /**
     * Load inject file content
     */
    public function getInject(string $type): string {
        if (!isset($this->config['inject'][$type]) || !$this->config['inject'][$type]) {
            return '';
        }
        
        $injectFile = $this->injectPath . '/' . $type . '.php';
        if (file_exists($injectFile)) {
            return file_get_contents($injectFile);
        }
        
        return '';
    }
    
    /**
     * Render header template
     */
    public function renderHeader(array $data): string {
        $headerFile = $this->templatePath . '/header.php';
        if (!file_exists($headerFile)) {
            return '';
        }
        
        extract($data);
        ob_start();
        include $headerFile;
        return ob_get_clean();
    }
    
    /**
     * Render footer template
     */
    public function renderFooter(array $data): string {
        $footerFile = $this->templatePath . '/footer.php';
        if (!file_exists($footerFile)) {
            return '';
        }
        
        extract($data);
        ob_start();
        include $footerFile;
        return ob_get_clean();
    }
    
    /**
     * Render article template
     */
    public function renderArticle(array $data): string {
        $articleFile = $this->templatePath . '/article.php';
        if (!file_exists($articleFile)) {
            return '';
        }
        
        extract($data);
        ob_start();
        include $articleFile;
        return ob_get_clean();
    }
    
    /**
     * Render image template
     */
    public function renderImage(array $data): string {
        $imageFile = $this->templatePath . '/image.php';
        if (!file_exists($imageFile)) {
            return '';
        }
        
        extract($data);
        ob_start();
        include $imageFile;
        return ob_get_clean();
    }
    
    /**
     * Full HTML render with all components
     */
    public function render(array $postData, array $relatedLinks = [], string $baseUrl = '', string $fileName = ''): string {
        // Parse JSON fields - support both 'articles' and 'content' columns
        $images = Util::parseJson($postData['images'] ?? null, []);
        $articles = Util::parseJson($postData['articles'] ?? ($postData['content'] ?? null), []);
        
        // Clean article text
        $articles = array_map([Util::class, 'cleanArticleText'], $articles);
        $articles = array_filter($articles, function($a) { return !empty($a); });
        
        // Get keyword first (needed for Bing URL generation)
        $keyword = Util::capitalizeWords($postData['keyword'] ?? 'No Title');
        
        // Generate Bing thumbnail URL - this is the PRIMARY image for schema
        $bingImageUrl = Util::generateBingImageUrl($keyword);
        
        // Get main image from database (for display, not for schema)
        $mainImage = '';
        if (!empty($images)) {
            $firstImage = $images[0];
            $imgUrl = $firstImage['url'] ?? ($firstImage['image'] ?? ($firstImage['src'] ?? ''));
            if (Util::isValidUrl($imgUrl)) {
                $mainImage = $imgUrl;
            }
        }
        
        // Get description
        $description = '';
        if (!empty($articles)) {
            $fullText = implode(' ', $articles);
            $firstSentence = Util::getFirstSentence($fullText);
            $description = Util::truncateText($firstSentence, 160);
        }
        
        // Generate canonical URL
        $canonicalUrl = '';
        if (!empty($baseUrl) && !empty($fileName)) {
            $canonicalUrl = rtrim($baseUrl, '/') . '/' . $fileName;
        }
        
        // Generate LD+JSON - use Bing thumbnail as PRIMARY image URL
        $ldJson = [
            '@context' => 'https://schema.org/',
            '@type' => 'Article',
            'author' => [
                '@type' => 'Person',
                'name' => 'Anggi',
                'url' => !empty($baseUrl) ? rtrim($baseUrl, '/') . '/about-us.html' : 'about-us.html'
            ],
            'headline' => $keyword,
            'description' => $description,
            'image' => $bingImageUrl, // Bing thumbnail as primary
            'publisher' => [
                '@type' => 'Organization',
                'name' => 'Anggi Ideas',
                'logo' => [
                    '@type' => 'ImageObject',
                    'url' => $bingImageUrl, // Bing thumbnail as primary
                    'width' => 250,
                    'height' => 250
                ]
            ]
        ];
        
        // Combine articles into groups of 3
        $combinedArticles = [];
        $fullText = implode(' ', $articles);
        $sentences = preg_split('/(?<=\.)\s+/', $fullText, -1, PREG_SPLIT_NO_EMPTY);
        for ($i = 0; $i < count($sentences); $i += 3) {
            $combinedArticles[] = implode(' ', array_slice($sentences, $i, 3));
        }
        
        // Prepare template data
        $templateData = [
            'keyword' => $keyword,
            'description' => $description,
            'mainImage' => $mainImage,
            'images' => $images,
            'articles' => $combinedArticles,
            'relatedLinks' => $relatedLinks,
            'ldJson' => $ldJson,
            'canonicalUrl' => $canonicalUrl,
            'config' => $this->config,
            'inject' => [
                'header' => $this->getInject('header'),
                'footer' => $this->getInject('footer'),
                'adsinarticle' => $this->getInject('adsinarticle')
            ]
        ];
        
        // Build HTML content
        $html = $this->renderHeader($templateData);
        
        // Add ads in article if configured
        if ($templateData['inject']['adsinarticle']) {
            $html .= $templateData['inject']['adsinarticle'];
        }
        
        // Interleave articles and images
        $articleIndex = 0;
        $imageIndex = 0;
        $h2Count = 0;
        
        while ($articleIndex < count($combinedArticles) || $imageIndex < count($images)) {
            // Add article with optional heading
            if ($articleIndex < count($combinedArticles)) {
                if (in_array($articleIndex, [2, 4, 6]) && isset($images[$imageIndex]) && !empty($images[$imageIndex]['title'])) {
                    $h2Title = Util::cleanArticleText($images[$imageIndex]['title']);
                    $html .= "<h2 class=\"text-2xl font-semibold mt-8 mb-4\">{$h2Title}</h2>";
                    $h2Count++;
                    
                    if ($h2Count === 3 && $templateData['inject']['adsinarticle']) {
                        $html .= $templateData['inject']['adsinarticle'];
                    }
                }
                
                $articleContent = $combinedArticles[$articleIndex];
                $html .= "<p class=\"mb-6 leading-relaxed\">{$articleContent}</p>";
                $articleIndex++;
            }
            
            // Add image
            if ($imageIndex < count($images)) {
                $img = $images[$imageIndex];
                $imgTitle = $img['title'] ?? $keyword;
                // Support both 'url', 'image', and 'src' field names
                $imgSrc = $img['url'] ?? ($img['image'] ?? ($img['src'] ?? ''));
                $imgDomain = $img['domain'] ?? '';
                
                // Validate image URL
                if (Util::isValidUrl($imgSrc)) {
                    $html .= $this->renderImage([
                        'title' => $imgTitle,
                        'src' => $imgSrc,
                        'domain' => $imgDomain,
                        'keyword' => $keyword
                    ]);
                }
                
                $imageIndex++;
            }
        }
        
        // Add related links
        if (!empty($relatedLinks)) {
            $html .= '<div class="mt-12 pt-6 border-t border-gray-300">';
            $html .= '<h3 class="text-xl font-semibold mb-4">Related Articles</h3>';
            $html .= '<ul class="list-disc list-inside space-y-2 text-blue-700">';
            
            foreach ($relatedLinks as $link) {
                $title = htmlspecialchars($link['title'], ENT_QUOTES, 'UTF-8');
                $href = htmlspecialchars($link['href'], ENT_QUOTES, 'UTF-8');
                $html .= "<li><a href=\"{$href}\" class=\"hover:underline\">{$title}</a></li>";
            }
            
            $html .= '</ul></div>';
        }
        
        $html .= $this->renderFooter($templateData);
        
        return $html;
    }
}
