<?php
/**
 * Main Exporter Class
 */

class Exporter {
    private $basePath;
    private $dataPath;
    private $exportPath;
    private $template;
    private $verbose;
    
    /**
     * Constructor
     */
    public function __construct(string $basePath, Template $template, bool $verbose = false) {
        $this->basePath = $basePath;
        $this->dataPath = $basePath . '/data';
        $this->exportPath = $basePath . '/export';
        $this->template = $template;
        $this->verbose = $verbose;
    }
    
    /**
     * Print message if verbose mode
     */
    public function log(string $message, string $type = 'INFO'): void {
        $timestamp = date('H:i:s');
        $prefix = "[{$timestamp}] [{$type}] ";
        
        // Windows compatibility - no ANSI colors
        if (PHP_OS === 'WINNT') {
            echo $prefix . $message . "\n";
        } else {
            $colors = [
                'INFO' => "\033[0;32m",
                'SUCCESS' => "\033[0;34m",
                'ERROR' => "\033[0;31m",
                'WARNING' => "\033[0;33m",
                'DEBUG' => "\033[0;36m"
            ];
            $color = $colors[$type] ?? "\033[0m";
            $reset = "\033[0m";
            echo $color . $prefix . $message . $reset . "\n";
        }
    }
    
    /**
     * Print progress bar
     */
    public function progressBar(int $current, int $total, int $width = 30): void {
        $percent = $total > 0 ? round(($current / $total) * 100) : 0;
        $filled = round($width * ($percent / 100));
        $empty = $width - $filled;
        
        $bar = str_repeat('=', $filled) . str_repeat('-', $empty);
        
        echo "\r[{$bar}] {$percent}% - {$current}/{$total} files";
        
        if ($current >= $total) {
            echo "\n";
        }
    }
    
    /**
     * Generate sitemap XML
     */
    public function generateSitemap(array $fileNames, string $baseUrl): string {
        if (!empty($baseUrl) && substr($baseUrl, -1) !== '/') {
            $baseUrl .= '/';
        }
        
        $urls = '';
        foreach ($fileNames as $fileName) {
            $urls .= "\n  <url>\n    <loc>" . $baseUrl . $fileName . "</loc>\n  </url>";
        }
        
        return "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<urlset xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\">" . $urls . "\n</urlset>";
    }
    
    /**
     * Process a single SQLite file
     */
    public function processSqliteFile(string $dbFileName, int $splitSize, string $baseUrl): array {
        $this->log("Processing SQLite file: " . $dbFileName);
        
        $dbPath = $this->dataPath . '/' . $dbFileName;
        
        try {
            $db = new Database($dbPath);
        } catch (Exception $e) {
            $this->log("Error opening database: " . $e->getMessage(), 'ERROR');
            return ['files' => 0, 'folders' => 0, 'error' => true];
        }
        
        // Check if posts table exists
        if (!$db->tableExists('posts') && !$db->tableExists('articles')) {
            $this->log("Table 'posts' or 'articles' not found in " . $dbFileName, 'WARNING');
            return ['files' => 0, 'folders' => 0, 'error' => false];
        }
        
        // Determine table name
        $tableName = $db->tableExists('posts') ? 'posts' : 'articles';
        
        // Get all posts
        $rows = $db->queryAll('SELECT * FROM ' . $tableName);
        
        if (empty($rows)) {
            $this->log("No posts found in " . $dbFileName, 'WARNING');
            return ['files' => 0, 'folders' => 0, 'error' => false];
        }
        
        // Filter valid posts (with articles and images as non-empty JSON arrays)
        $validPosts = array_filter($rows, function($row) {
            $content = $row['content'] ?? $row['articles'] ?? null;
            $images = $row['images'] ?? null;
            
            // Check if content is valid JSON array (not empty "[]")
            $contentValid = false;
            if (!empty($content) && $content !== '[]') {
                $decoded = json_decode($content, true);
                $contentValid = is_array($decoded) && !empty($decoded);
            }
            
            // Check if images is valid JSON array (not empty "[]")
            $imagesValid = false;
            if (!empty($images) && $images !== '[]') {
                $decoded = json_decode($images, true);
                $imagesValid = is_array($decoded) && !empty($decoded);
            }
            
            return $contentValid && $imagesValid;
        });
        
        if (empty($validPosts)) {
            $this->log("No valid posts (with articles and images) in " . $dbFileName, 'WARNING');
            return ['files' => 0, 'folders' => 0, 'error' => false];
        }
        
        $this->log("Found " . count($validPosts) . " valid posts in " . $dbFileName);
        
        // Re-index array
        $validPosts = array_values($validPosts);
        
        // Split into chunks
        $chunks = array_chunk($validPosts, $splitSize);
        $totalFiles = 0;
        $totalFolders = count($chunks);
        
        foreach ($chunks as $chunkIndex => $chunk) {
            $folderName = 'folder_' . ($chunkIndex + 1);
            $baseFolderName = preg_replace('/\.sqlite$/i', '', $dbFileName);
            $folderPath = $this->exportPath . '/' . $baseFolderName . '/' . $folderName;
            
            // Create folder if not exists
            if (!is_dir($folderPath)) {
                mkdir($folderPath, 0755, true);
            }
            
            // Generate about-us.html for this folder
            $this->generateAboutUs($folderPath, $chunkIndex);
            
            $this->log("Exporting " . $folderName . "/ (" . count($chunk) . " files)");
            
            // Prepare file info
            $fileInfos = [];
            foreach ($chunk as $index => $row) {
                $slug = !empty(trim($row['slug'] ?? '')) 
                    ? trim($row['slug']) 
                    : Util::createSafeSlug($row['keyword'] ?? 'post');
                
                $fileName = $slug . '.html';
                $fileInfos[] = ['row' => $row, 'fileName' => $fileName];
            }
            
            // Generate all file names for related links
            $allFileNames = array_column($fileInfos, 'fileName');
            
            // Export each file
            $chunkFiles = 0;
            foreach ($fileInfos as $index => $info) {
                // Generate related links (exclude current file)
                $relatedLinks = [];
                foreach ($allFileNames as $fileName) {
                    if ($fileName !== $info['fileName']) {
                        $title = preg_replace('/\.html$/i', '', $fileName);
                        $title = str_replace('-', ' ', $title);
                        $relatedLinks[] = [
                            'href' => $fileName,
                            'title' => ucwords($title)
                        ];
                    }
                }
                // Limit to 5 related links
                $relatedLinks = array_slice($relatedLinks, 0, 5);
                
                // Render HTML
                $htmlContent = $this->template->render($info['row'], $relatedLinks, $baseUrl, $info['fileName']);
                
                // Write file
                $filePath = $folderPath . '/' . $info['fileName'];
                file_put_contents($filePath, $htmlContent);
                
                $chunkFiles++;
                
                // Show progress
                if ($this->verbose) {
                    $this->progressBar($chunkFiles, count($chunk));
                }
            }
            
            // Generate sitemap
            $sitemap = $this->generateSitemap($allFileNames, $baseUrl);
            file_put_contents($folderPath . '/sitemap.xml', $sitemap);
            
            $totalFiles += $chunkFiles;
            $this->log("Created {$chunkFiles} HTML files in {$folderName}/");
        }
        
        $this->log("Processed " . $dbFileName . ": created " . $totalFiles . " HTML files in " . $totalFolders . " folders.", 'SUCCESS');
        
        return [
            'files' => $totalFiles,
            'folders' => $totalFolders,
            'error' => false
        ];
    }
    
    /**
     * Get spintax-based about us content (approximately 250 words)
     * Each folder gets unique content using spintax variations
     */
    private function getAboutUsContent(int $folderIndex): string {
        $names = "{James|William|Thomas|Robert|Michael|David|Richard|Joseph|Charles|Christopher|Daniel|Matthew|Anthony|Mark|Donald|Steven|Paul|Andrew|Joshua|Kenneth|Kevin|Brian|George|Timothy|Ronald|Edward|Jason|Jeffrey|Ryan|Jacob|Gary|Nicholas|Eric|Jonathan|Stephen|Larry|Justin|Scott|Brandon|Benjamin|Samuel|Raymond|Gregory|Frank|Alexander|Patrick|Jack|Dennis|Jerry|Tyler|Aaron|Jose|Adam|Nathan|Henry|Douglas|Zachary|Peter|Kyle|Noah|Ethan|Jeremy|Walter|Christian|Keith|Roger|Terry|Austin|Sean|Gerald|Carl|Harold|Dylan|Arthur|Lawrence|Jordan|Jesse|Bryan|Billy|Bruce|Gabriel|Joe|Logan|Albert|Willie|Alan|Eugene|Russell|Vincent|Bobby|Philip|Roy|Louis|Johnny|Howard|Phillip|Randy|Eugene|Ralph|Ellen|Jessica|Jennifer|Lisa|Sarah|Michelle|Amanda|Stephanie|Jessica|Heather|Megan|Rebecca|Kimberly|Emily|Elizabeth|Sarah|Ashley|Kimberly|Stephanie|Nicole|Lisa|Christine|Deborah|Michelle|Kimberly|Karen|Nancy|Betty|Helen|Sandra|Donna|Carol|Ruth|Sharon|Michelle|Jennifer|Sandra|Amanda|Melissa|Deborah|Stephanie|Rebecca|Sharon|Lisa|Patricia|Kimberly|Jennifer}";
        
        $spintax = ""
            . "<p>Hello! I'm " . $names . ", {a passionate writer|an avid blogger|a curious content creator|an enthusiastic storyteller|a dedicated creator|a lifelong learner} who has spent over {five|seven|ten|twelve|fifteen} years {exploring the world of digital content|sharing my thoughts and ideas|documenting my journey|building an online presence|connecting with readers|creating meaningful content}.</p>\n\n"
            . "<p>My journey began {when I discovered my love for writing during college|when I started my first blog back in 2010|when I decided to share my experiences online|after years of keeping journals and notebooks|during a pivotal moment of self-discovery|when I realized I had stories worth sharing}. I quickly found that {writing is my way of connecting with others|sharing knowledge brings me joy|the creative process is incredibly rewarding|every day presents new opportunities to learn|putting thoughts into words can inspire and educate|building a community of like-minded people is fulfilling}.</p>\n\n"
            . "<p>Over the years, I have {honed my craft through countless articles|built relationships with an amazing audience|explored various topics and niches|learned the art of storytelling|developed a unique voice and perspective|discovered the power of authentic content}. What started as {a personal hobby|a creative outlet|a way to document my thoughts|a small project|a simple blog|a passion project} has grown into {something I truly cherish|a meaningful pursuit|a source of inspiration|a journey of growth|a platform for sharing|a community of readers}.</p>\n\n"
            . "<p>I believe that {every person has a story worth telling|great content comes from authenticity|learning never stops|sharing is caring|creativity knows no bounds|passion drives excellence}. Through my work, I strive to {inspire and motivate others|provide valuable insights|spark meaningful conversations|make a positive impact|share knowledge and experiences|connect with people from all walks of life}.</p>\n\n"
            . "<p>When I'm not {writing|creating|blogging|exploring|researching}, you can find me {enjoying a good cup of coffee|reading a fascinating book|traveling to new places|spending time with family and friends|exploring nature|learning new skills|hiking and adventure|photography and visual arts|playing guitar and music}. I believe that {balance is key to creativity|life is full of adventures|staying curious keeps us young|every experience adds to our story|we never stop growing}.</p>\n\n"
            . "<p>Thank you for {being part of this journey|visiting my corner of the internet|joining me here|stopping by|connecting with me|reading my content}. I am {truly grateful|deeply appreciate|always humbled|sincerely thankful} for {your time and attention|the community we've built|everyone who reads and connects|this opportunity to share|your support and encouragement}. Feel free to {reach out|connect with me|share your thoughts|drop me a message|explore more content} anytime!</p>\n\n"
            . "<p>{Warm regards|With appreciation|Sincerely|Best wishes|Kind regards|Much love},</p>\n\n"
            . "<p class=\"font-semibold\">" . $names . "</p>\n"
            . "<p class=\"text-sm text-gray-600\">{Content Creator|Blogger|Content Creator|Blogger|Content Creator|Blogger}</p>\n"
            . "";
        
        // Parse spintax to get unique content
        return Util::parseSpintax($spintax);
    }
    
    /**
     * Generate about-us.html for a folder
     */
    public function generateAboutUs(string $folderPath, int $folderIndex): void {
        $aboutUsContent = $this->getAboutUsContent($folderIndex);
        
        $aboutUsHtml = <<<HTML
<!doctype html>
<html lang="id">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>About Us</title>
  <meta name="description" content="Learn more about us and our passion for home improvement." />
  <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-blue-100 text-gray-900 pt-8">
  <div class="min-h-screen flex justify-center items-start px-4 py-8">
    <div class="bg-gray-50 container max-w-4xl rounded-md shadow-lg p-8">
      <h1 class="text-4xl font-bold mb-6">About Us</h1>
      <div class="space-y-6 leading-relaxed text-gray-700">
        {$aboutUsContent}
      </div>
      <div class="mt-8 pt-6 border-t border-gray-300">
        <a href="./" class="text-blue-700 hover:underline">← Back to Home</a>
      </div>
    </div>
  </div>
</body>
</html>
HTML;
        
        $filePath = $folderPath . '/about-us.html';
        file_put_contents($filePath, $aboutUsHtml);
        $this->log("Created about-us.html in folder: " . basename($folderPath));
    }
    
    /**
     * Main export function
     */
    public function export(int $splitSize, string $baseUrl): bool {
        $this->log("Starting HTML export...");
        $this->log("Template: " . $this->template->getName());
        $this->log("Split size: " . $splitSize);
        $this->log("Base URL: " . ($baseUrl ?: '(none)'));
        
        // Check data folder
        if (!is_dir($this->dataPath)) {
            $this->log("Data folder not found: " . $this->dataPath, 'ERROR');
            return false;
        }
        
        // Get SQLite files
        $sqliteFiles = array_filter(scandir($this->dataPath), function($file) {
            return preg_match('/\.sqlite$/i', $file);
        });
        
        if (empty($sqliteFiles)) {
            $this->log("No SQLite files found in data folder.", 'WARNING');
            return false;
        }
        
        $this->log("Found " . count($sqliteFiles) . " SQLite file(s)");
        
        // Ensure export folder exists
        if (!is_dir($this->exportPath)) {
            mkdir($this->exportPath, 0755, true);
        }
        
        // Process each file
        $totalFiles = 0;
        $totalFolders = 0;
        $hasErrors = false;
        
        foreach ($sqliteFiles as $dbFile) {
            $result = $this->processSqliteFile($dbFile, $splitSize, $baseUrl);
            $totalFiles += $result['files'];
            $totalFolders += $result['folders'];
            if ($result['error']) {
                $hasErrors = true;
            }
        }
        
        // Summary
        $this->log("========================================");
        $this->log("Export completed!", 'SUCCESS');
        $this->log("Total: " . $totalFiles . " HTML files in " . $totalFolders . " folders");
        $this->log("Output: " . $this->exportPath);
        $this->log("Export selesai", 'SUCCESS');
        
        return !$hasErrors;
    }
}
