const fs = require('fs');
const path = require('path');
const sqlite3 = require('sqlite3').verbose();
const util = require('util');

const fsPromises = fs.promises;

const injectFolder = path.join(__dirname, 'inject');
const headerInject = fs.existsSync(path.join(injectFolder, 'header.php'))
  ? fs.readFileSync(path.join(injectFolder, 'header.php'), 'utf8')
  : '';
const footerInject = fs.existsSync(path.join(injectFolder, 'footer.php'))
  ? fs.readFileSync(path.join(injectFolder, 'footer.php'), 'utf8')
  : '';
const adsInArticleInject = fs.existsSync(path.join(injectFolder, 'adsinarticle.php'))
  ? fs.readFileSync(path.join(injectFolder, 'adsinarticle.php'), 'utf8')
  : '';

function cleanArticleText(text) {
  if (!text) return '';
  try {
    return JSON.parse('"' + text.replace(/"/g, '\\\\"') + '"').trim();
  } catch {
    return text.trim();
  }
}

function capitalizeWords(str) {
  return str.replace(/\b\w/g, c => c.toUpperCase());
}

function generateHtmlContent(row, safeSlug, relatedLinks = []) {
  const keyword = row.keyword ? capitalizeWords(row.keyword) : 'No Title';

  let images = [];
  if (typeof row.images === 'string') {
    try {
      images = JSON.parse(row.images);
    } catch {
      images = [];
    }
  } else if (Array.isArray(row.images)) {
    images = row.images;
  }

  let mainImage = '';
  if (images.length > 0) {
    let imgUrl = images[0].image || images.src || '';
    if (imgUrl && !/^https?:\/\//i.test(imgUrl)) {
      if (global.baseUrl) {
        let base = global.baseUrl.replace(/^https?:\/\//i, '').replace(/\/$/, '');
        imgUrl = `${base}/${imgUrl.replace(/^\/+/, '')}`;
      }
    }
    mainImage = imgUrl;
  }

  let articles = [];
  if (typeof row.articles === 'string') {
    try {
      articles = JSON.parse(row.articles);
    } catch {
      articles = [];
    }
  } else if (Array.isArray(row.articles)) {
    articles = row.articles;
  }
  articles = articles.map(cleanArticleText).filter(a => a.length > 0);

  let description = '';
  if (articles.length > 0) {
    const firstSentence = articles.join(' ').split(/(?<=\.)\s+/)[0].trim();
    description = firstSentence.length > 160 ? firstSentence.slice(0, 157) + '...' : firstSentence;
  }

  const ldJson = {
    "@context": "https://schema.org/",
    "@type": "Article",
    "author": {
      "@type": "Person",
      "name": "Anggi",
      "url": "https://www.google.com/search?q=Anggi"
    },
    "headline": keyword,
    "description": description,
    "image": mainImage,
    "publisher": {
      "@type": "Organization",
      "name": "Anggi Ideas",
      "logo": {
        "@type": "ImageObject",
        "url": mainImage || "https://tse1.mm.bing.net/th?q=default+logo&w=250&h=250&c=7",
        "width": 250,
        "height": 250
      }
    }
  };

  const combinedArticles = [];
  const sentences = articles.join(' ').split(/(?<=\.)\s+/).filter(Boolean);
  for(let i=0; i<sentences.length; i+=3) {
    combinedArticles.push(sentences.slice(i, i+3).join(' '));
  }

  let content = `<!doctype html>
<html lang="id">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>${keyword}</title>
  <meta name="description" content="${description.replace(/"/g, '&quot;')}" />
  <link href="https://unpkg.com/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet" />
  ${headerInject}
  <script type="application/ld+json">${JSON.stringify(ldJson, null, 2)}</script>
</head>
<body class="bg-blue-100 text-gray-900 pt-8">
  <div class="min-h-screen flex justify-center items-start px-4 py-8">
    <div class="bg-gray-50 container max-w-4xl rounded-md shadow-lg p-8">
      <h1 class="text-4xl font-bold mb-6">${keyword}</h1>`;

  if (adsInArticleInject) {
    content += `<div class="mb-6">${adsInArticleInject}</div>`;
  }

  let articleIndex = 0;
  let imageIndex = 0;
  let h2Count = 0;

  while (articleIndex < combinedArticles.length || imageIndex < images.length) {
    if (articleIndex < combinedArticles.length) {
      if ([2,4,6].includes(articleIndex) && images[imageIndex] && images[imageIndex].title) {
        content += `<h2 class="text-2xl font-semibold mt-8 mb-4">${cleanArticleText(images[imageIndex].title)}</h2>`;
        h2Count ++;
        if (h2Count === 3 && adsInArticleInject) {
          content += `<div class="mb-6">${adsInArticleInject}</div>`;
        }
      }
      content += `<p class="mb-6 leading-relaxed">${combinedArticles[articleIndex]}</p>`;
      articleIndex++;
    }
    if (imageIndex < images.length) {
      const img = images[imageIndex];
      const imgTitle = img.title || keyword;
      const imgSrc = img.image || '';
      const domain = img.domain || '';
      content += `<div class="mb-8">
  <img src="${imgSrc}" alt="${imgTitle}" title="${imgTitle}" class="rounded shadow-md w-full max-w-full" />
  <p class="text-sm text-gray-500 mt-2">${domain}</p>
</div>`;
      imageIndex++;
    }
  }

  if (relatedLinks.length > 0) {
    content += `<div class="mt-12 pt-6 border-t border-gray-300">
  <h3 class="text-xl font-semibold mb-4">Related Articles</h3>
  <ul class="list-disc list-inside space-y-2 text-blue-700">`;
    relatedLinks.forEach(({title, href})=>{
      content += `<li><a href="${href}" class="hover:underline">${title}</a></li>`;
    });
    content += `</ul></div>`;
  }

  content += `
    ${footerInject}
    </div>
  </div>
</body>
</html>`;

  return content;
}

function generateSitemapXml(fileNames, baseUrl) {
  if (typeof baseUrl !== 'string') baseUrl = '';
  if (baseUrl.length > 0 && !baseUrl.endsWith('/')) baseUrl += '/';

  const urlSet = fileNames.map(fileName => `
  <url>
    <loc>${baseUrl}${fileName}</loc>
  </url>`).join('\n');

  return `<?xml version="1.0" encoding="UTF-8"?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">
${urlSet}
</urlset>`;
}

async function processSqliteFile(dbFileName, splitSize, baseUrl) {
  const dbPath = path.join(__dirname, 'data', dbFileName);
  const db = new sqlite3.Database(dbPath, sqlite3.OPEN_READONLY);
  const dbAll = util.promisify(db.all).bind(db);
  let rows;
  try {
    rows = await dbAll('SELECT * FROM posts');
  } catch (err) {
    console.error(`Error querying posts in ${dbFileName}:`, err.message);
    db.close();
    return;
  }
  if (!rows || rows.length === 0) {
    console.log(`No posts in ${dbFileName}`);
    db.close();
    return;
  }
  rows = rows.filter(r=>r.articles != null && r.images != null);
  if (rows.length === 0) {
    console.log(`No valid posts in ${dbFileName} after filtering`);
    db.close();
    return;
  }

  const chunks = [];
  for(let i=0; i<rows.length; i+=splitSize) {
    chunks.push(rows.slice(i, i+splitSize));
  }

  let totalFiles = 0;
  for(let idx=0; idx<chunks.length; idx++) {
    const folderName = `folder_${idx+1}`;
    const baseFolderName = dbFileName.replace(/\.sqlite$/i, '');
    const folderPath = path.join(__dirname, 'export', baseFolderName, folderName);
    if (!fs.existsSync(folderPath)) {
      fs.mkdirSync(folderPath, {recursive:true});
    }
    
    const fileInfos = chunks[idx].map((row,i)=>{
      let safeSlug = row.slug ? row.slug.trim() : '';
      if (!safeSlug) {
        safeSlug = row.keyword ? row.keyword.trim().toLowerCase().replace(/\s+/g,'-') : `post_${i+1}`;
      }
      safeSlug = safeSlug.replace(/[^a-z0-9\-]/gi,'').toLowerCase();
      if (!safeSlug) safeSlug = `post_${i+1}`;
      const fileName = `${safeSlug}.html`;
      return {row, fileName};
    });

    const fileNames = fileInfos.map(info => info.fileName);

    const concurrency = 10;
    for(let i=0; i<fileInfos.length; i+=concurrency) {
      const batch = fileInfos.slice(i, i+concurrency);
      await Promise.all(batch.map(async ({row, fileName}) => {
        const relatedLinks = fileNames
          .filter(f => f !== fileName)
          .map(f => ({
            href: f,
            title: f.replace(/\.html$/i, '').replace(/-/g, ' ')
          }))
          .slice(0,5);

        const htmlContent = generateHtmlContent(row, '', relatedLinks);
        const filePath = path.join(folderPath, fileName);
        await fsPromises.writeFile(filePath, htmlContent, 'utf8');
      }));
    }

    const sitemap = generateSitemapXml(fileNames, baseUrl);
    await fsPromises.writeFile(path.join(folderPath, 'sitemap.xml'), sitemap, 'utf8');

    totalFiles += fileNames.length;
  }

  db.close();
  console.log(`Processed ${dbFileName}: created ${totalFiles} HTML files in ${chunks.length} folders.`);
}

async function main() {
  const args = process.argv.slice(2);
  if (args.length < 2) {
    console.error('Usage: node export_sqlite_to_html.js <splitSize> <baseUrl>');
    process.exit(1);
  }
  
  const splitSize = parseInt(args[0],10);
  global.baseUrl = String(args[1] || '');
  
  if (isNaN(splitSize) || splitSize <= 0) {
    console.error('splitSize must be a positive integer');
    process.exit(1);
  }

  const dataFolder = path.join(__dirname, 'data');
  let files = [];
  try {
    files = await fsPromises.readdir(dataFolder);
    files = files.filter(f => /\.sqlite$/i.test(f));
  } catch(e) {
    console.error('Failed to read data folder:', e.message);
    process.exit(1);
  }
  if (files.length === 0) {
    console.log('No SQLite files found in data folder.');
    return;
  }

  for (const fileName of files) {
    console.log(`Processing SQLite file: ${fileName}`);
    await processSqliteFile(fileName, splitSize, global.baseUrl);
  }
  console.log('All files processed.');
}

main();
