#!/usr/bin/env php
<?php
/**
 * PHP CLI HTML Exporter
 * 
 * Export SQLite data to HTML files with template support.
 * 
 * Usage:
 *   php export_html.php <splitSize> <baseUrl> [--template=<name>] [--verbose]
 * 
 * Arguments:
 *   splitSize   Number of posts per folder (default: 50)
 *   baseUrl     Base URL for sitemap generation (optional)
 * 
 * Options:
 *   --template  Template name to use (default: default)
 *   --verbose   Show detailed progress
 *   --help      Show this help message
 * 
 * Examples:
 *   php export_html.php 50 "https://example.com"
 *   php export_html.php 100 "https://example.com" --template=default --verbose
 */

// Increase memory limit for large databases
ini_set('memory_limit', '1024M');

// Define base path
define('BASE_PATH', __DIR__);

// Autoload classes
require_once BASE_PATH . '/lib/Util.php';
require_once BASE_PATH . '/lib/Database.php';
require_once BASE_PATH . '/lib/Template.php';
require_once BASE_PATH . '/lib/Exporter.php';

/**
 * Show help message
 */
function showHelp(): void {
    echo <<<HELP
PHP CLI HTML Exporter
=====================

Usage:
  php export_html.php <splitSize> <baseUrl> [--template=<name>] [--verbose]

Arguments:
  splitSize   Number of posts per folder (default: 50)
  baseUrl     Base URL for sitemap generation (optional)

Options:
  --template   Template name to use (default: default)
  --verbose    Show detailed progress
  --help       Show this help message
  --list       List available templates

Examples:
  php export_html.php 50 "https://example.com"
  php export_html.php 100 "https://example.com" --template=default --verbose
  php export_html.php --list

Available Templates:
  default     - Tailwind CSS based template (recommended)
  gallery     - Image-focused gallery layout
  simple      - Minimal HTML template

HELP;
}

/**
 * List available templates
 */
function listTemplates(): void {
    echo "\nAvailable Templates:\n";
    echo "====================\n\n";
    
    $templates = Template::getAvailableTemplates(BASE_PATH);
    
    if (empty($templates)) {
        echo "No templates found in templates/ folder.\n";
        return;
    }
    
    foreach ($templates as $name => $description) {
        echo "  - {$name}";
        if ($name !== $description) {
            echo ": {$description}";
        }
        echo "\n";
    }
    
    echo "\n";
}

/**
 * Parse command line arguments
 */
function parseArgs(array $argv): array {
    $args = [
        'splitSize' => 50,
        'baseUrl' => '',
        'template' => 'default',
        'verbose' => false,
        'help' => false,
        'list' => false
    ];
    
    foreach ($argv as $arg) {
        // Skip script name
        if ($arg === $argv[0]) continue;
        
        // Help flag
        if ($arg === '--help' || $arg === '-h') {
            $args['help'] = true;
            continue;
        }
        
        // List flag
        if ($arg === '--list') {
            $args['list'] = true;
            continue;
        }
        
        // Verbose flag
        if ($arg === '--verbose' || $arg === '-v') {
            $args['verbose'] = true;
            continue;
        }
        
        // Template option
        if (strpos($arg, '--template=') === 0) {
            $args['template'] = substr($arg, 11);
            continue;
        }
        
        // Template short option
        if (strpos($arg, '-t=') === 0) {
            $args['template'] = substr($arg, 3);
            continue;
        }
        
        // Numeric argument (splitSize)
        if (is_numeric($arg)) {
            $args['splitSize'] = (int)$arg;
            continue;
        }
        
        // Non-numeric, non-option argument (baseUrl)
        if (strpos($arg, '-') !== 0) {
            $args['baseUrl'] = $arg;
            continue;
        }
    }
    
    return $args;
}

/**
 * Main execution
 */
function main(array $argv): int {
    $args = parseArgs($argv);
    
    // Show help
    if ($args['help']) {
        showHelp();
        return 0;
    }
    
    // List templates
    if ($args['list']) {
        listTemplates();
        return 0;
    }
    
    // Validate arguments
    if ($args['splitSize'] <= 0) {
        echo "Error: splitSize must be a positive integer.\n";
        echo "Use --help for usage information.\n";
        return 1;
    }
    
    // Load template
    $template = new Template($args['template'], BASE_PATH);
    
    if (!$template->isValid()) {
        echo "Error: Template '" . $args['template'] . "' is not valid or missing required files.\n";
        echo "Use --list to see available templates.\n";
        return 1;
    }
    
    // Create exporter
    $exporter = new Exporter(BASE_PATH, $template, $args['verbose']);
    
    // Run export
    $success = $exporter->export($args['splitSize'], $args['baseUrl']);
    
    return $success ? 0 : 1;
}

// Run main function
exit(main($argv));
