# PHP CLI HTML Exporter

A PHP command-line tool to export SQLite data to HTML files with template support.

## Features

- **PHP CLI** - No Node.js required
- **Visual Progress** - Real-time export status in CMD
- **Multiple Templates** - Switch between different HTML layouts
- **Bing Image URLs** - Automatic Bing image URL generation for image objects
- **URL Validation** - Only valid https:// URLs are used
- **Sitemap Generation** - Automatic sitemap.xml creation
- **Related Links** - Auto-generated related articles section
- **Inject Support** - Compatible with existing inject files

## Requirements

- PHP 7.4 or higher
- PHP SQLite3 extension enabled
- SQLite3 library

## Installation

1. Ensure PHP is installed and available in PATH
2. This tool requires no additional dependencies

## Usage

### Basic Usage

```bash
php export_html.php <splitSize> <baseUrl>
```

### With Options

```bash
php export_html.php 50 "https://example.com" --template=default --verbose
```

### Command Line Options

| Option | Description |
|--------|-------------|
| `splitSize` | Number of posts per folder (default: 50) |
| `baseUrl` | Base URL for sitemap generation (optional) |
| `--template=<name>` | Template to use (default: default) |
| `--verbose` | Show detailed progress |
| `--help` | Show help message |
| `--list` | List available templates |

### Windows Batch File

You can also use the batch file for easier execution:

```bash
export_html.bat 50 "https://example.com"
export_html.bat --help
```

## Available Templates

### default (Default)
Tailwind CSS based template with responsive design and inject support.

```bash
php export_html.php 50 "https://example.com" --template=default
```

### gallery
Image-focused gallery layout with large featured images.

```bash
php export_html.php 50 "https://example.com" --template=gallery
```

### simple
Minimal HTML template without external CSS frameworks.

```bash
php export_html.php 50 "https://example.com" --template=simple
```

## Creating Custom Templates

To create a new template, follow these steps:

1. Create a new folder in `templates/` directory
2. Add the following required files:

### 1. config.php
Template configuration file:

```php
<?php
return [
    'name' => 'My Custom Template',
    'description' => 'Description of your template',
    'version' => '1.0',
    'author' => 'Your Name',
    'css' => [
        'https://example.com/style.css'
    ],
    'inject' => [
        'header' => true,
        'footer' => true,
        'adsinarticle' => true
    ],
    'settings' => [
        'container_max_width' => '4xl',
        'show_related_links' => true,
        'related_links_count' => 5
    ]
];
```

### 2. header.php
HTML header section with variables:

```php
<!doctype html>
<html lang="id">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title><?php echo $keyword; ?></title>
  <meta name="description" content="<?php echo $description; ?>" />
  <?php foreach ($config['css'] as $css): ?>
  <link href="<?php echo $css; ?>" rel="stylesheet" />
  <?php endforeach; ?>
  <?php if (!empty($inject['header'])): ?>
  <?php echo $inject['header']; ?>
  <?php endif; ?>
  <script type="application/ld+json"><?php echo json_encode($ldJson); ?></script>
</head>
<body>
  <div class="container">
    <h1><?php echo $keyword; ?></h1>
```

### 3. footer.php
HTML footer section:

```php
    <?php if (!empty($inject['footer'])): ?>
    <?php echo $inject['footer']; ?>
    <?php endif; ?>
  </div>
</body>
</html>
```

### 4. article.php
Article paragraph template:

```php
<p class="mb-4"><?php echo $content; ?></p>
```

### 5. image.php
Image display template:

```php
<div class="mb-4">
  <img src="<?php echo $src; ?>" alt="<?php echo $title; ?>" title="<?php echo $title; ?>" />
  <?php if (!empty($domain)): ?>
  <p class="text-sm text-gray-500"><?php echo $domain; ?></p>
  <?php endif; ?>
</div>
```

### Available Template Variables

| Variable | Description |
|----------|-------------|
| `$keyword` | Article title (capitalized) |
| `$description` | Meta description (truncated to 160 chars) |
| `$mainImage` | Primary image URL |
| `$images` | Array of image objects |
| `$articles` | Array of article paragraphs (cleaned) |
| `$relatedLinks` | Array of related article links |
| `$ldJson` | Schema.org structured data (array) |
| `$config` | Template configuration array |
| `$inject` | Inject content array (header, footer, adsinarticle) |

## Image URL Handling

The exporter automatically handles image URLs:

1. **Valid URLs** - URLs starting with `https://` are used directly
2. **Invalid URLs** - Bing image URL is generated using the keyword
3. **Bing Format** - `https://tse1.mm.bing.net/th?q=keyword&w=250&h=250&c=7`

## Output Structure

```
export/
├── [database_name]/
│   ├── folder_1/
│   │   ├── article-1.html
│   │   ├── article-2.html
│   │   └── sitemap.xml
│   ├── folder_2/
│   │   └── ...
│   └── folder_3/
│       └── ...
```

## Example Output

```
[18:00:00] [INFO] Starting HTML export...
[18:00:00] [INFO] Template: Default Template
[18:00:00] [INFO] Split size: 50
[18:00:00] [INFO] Base URL: https://example.com
[18:00:00] [INFO] Found 1 SQLite file(s)
[18:00:00] [INFO] Processing SQLite file: Home1.sqlite
[18:00:00] [INFO] Found 150 valid posts in Home1.sqlite
[18:00:00] [INFO] Exporting folder_1/ (50 files)
[18:00:01] [INFO] Created 50 HTML files in folder_1/
[18:00:02] [INFO] Exporting folder_2/ (50 files)
[18:00:03] [INFO] Created 50 HTML files in folder_2/
[18:00:04] [INFO] Exporting folder_3/ (50 files)
[18:00:05] [INFO] Created 50 HTML files in folder_3/
[18:00:05] [INFO] Processed Home1.sqlite: created 150 HTML files in 3 folders.
========================================
[18:00:05] [INFO] Export completed!
[18:00:05] [INFO] Total: 150 HTML files in 3 folders
[18:00:05] [INFO] Output: export/
[18:00:05] [SUCCESS] Export selesai
```

## Troubleshooting

### "SQLite3 not found" error
Make sure PHP SQLite3 extension is enabled in php.ini:
```ini
extension=sqlite3
```

### "Template not found" error
Check that the template folder exists in `templates/` directory and contains all required files.

### No HTML files generated
Verify that SQLite files exist in the `data/` directory and contain valid posts with articles and images.

## License

MIT License
